//======================== FORM ASPECT RATIO CONTROL ===========================
//
// Component to control the aspect ratio of either a form or a form's client
// area.  This component came about because I was loading bitmaps into a TImage
// on a form with the TImage set to alClient and Stretch to true.  This meant the
// bitmap was displayed at the scale that the form was sized to, but meant that
// the image became distorted if the aspect ratio of the original bitmap was
// not maintained - not an easy thing to do manually.
//
// The component works by adding a new WndProc and intecepting the (win 32)
// WM_SIZING message.  This messgae is send repeatedly while a window
// is being resize by dragging a corner or side.  The component calculates the
// "other" side from the one being dragged forcing a given aspect ratio to be
// maintained.  The component also intercepts the WM_GETMINMAXINFO message
// to ensure that the aspect ratio is maintained when the form is zoomed to
// full size.
//
// NB This component will "fight" with to TMinMax component.  The second of this
// and TMinMax placed on a form will win.
//
// Version 1.10
// Grahame Marsh 19 January 1997
//
// Freeware - you get it for free, I take nothing, I make no promises!
//
// Please feel free to contact me: grahame.s.marsh@corp.courtaulds.co.uk

unit
  Aspect;

interface

uses
  SysUtils, Messages, Classes, Forms, Windows, LibConst;

// side adjust constants, used as values for the side parameter in the
// EstablishAspectRatio procedure
const
  saLeft   = WMSZ_Top;
  saTop    = WMSZ_Left;
  saBottom = WMSZ_Right;
  saRight  = WMSZ_Bottom;

type
  TModClientEvent = procedure (Sender : TObject; var X, Y : integer) of object;
  TBeforeResizingEvent = procedure (Sender : TObject; var R : TRect) of object;

  TAspect = class(TComponent)
  private
    FParent : THandle;         // these are used to subclass the form
    FOldDefWndProc,
    FNewDefWndProc : pointer;

    FOnModifyClient : TModClientEvent;
    FOnBeforeResizing : TBeforeResizingEvent;

    FAspectX,                  // X aspect and YAspect are relatively unimportant
    FAspectY : integer;        // it is the ratio that matters.

    FActive,                   // is the component effect on or off

    FClient : boolean;         // are we controlling the client or whole ratio
    procedure NewDefWndProc (var Msg : TMessage);
  protected
  public
    constructor Create (AOwner : TComponent); override;
    destructor  Destroy; override;
    procedure   Loaded; override;
    procedure EstablishAspectRatio (Side : word; var Rect : TRect); virtual;
  published
// activate the aspect ratio control
    property Active : boolean read FActive write FActive default true;
// set the X part of the aspect ratio
    property AspectX : integer read FAspectX write FAspectX default 1;
// set the Y part of the aspect ratio
    property AspectY : integer read FAspectY write FAspectY default 1;
// does the ratio apply to the client area (T) or to the whole form area (F)?
    property Client : boolean read FClient write FClient default true;

    property OnModifyClient : TModClientEvent read FOnModifyClient write FOnModifyClient;
    property OnBeforeResizing : TBeforeResizingEvent read FOnBeforeResizing write FOnBeforeResizing;
  end;

procedure Register;

implementation

constructor TAspect.Create (AOwner : TComponent);
begin
  inherited Create (AOwner);
// prepare to subclass the window
  FParent := (AOwner as TForm).Handle;
  FNewDefWndProc := MakeObjectInstance (NewDefWndProc);
// default conditions
  FActive := true;
  FClient := true;
  FAspectX := 1;
  FAspectY := 1
end;

destructor TAspect.Destroy;
begin
// remove events
  FOnModifyClient := nil;
  FOnBeforeResizing := nil;
// take our WndProc out of use
  SetWindowLong (FParent, GWL_WndProc, longint(FOldDefWndProc));
// chuck the object instance
  FreeObjectInstance (FNewDefWndProc);
  inherited Destroy
end;

procedure TAspect.Loaded;
begin
  inherited Loaded;
// put our WndProc into use
  FOldDefWndProc := pointer(SetWindowLong (FParent, GWL_WndProc, longint(FNewDefWndProc)))
end;

// Take a rect and force an aspect ratio on it, the pulled by value
// effects which side is adjusted to make the aspect ratio work.
// saXXXX constants are used.

procedure TAspect.EstablishAspectRatio (Side : word; var Rect : TRect);
var
  ClientXmod,
  ClientYmod : integer;
begin
  if FClient then
    with (Owner as TForm) do
    begin
      ClientYmod := Height - ClientHeight;
      ClientXMod := Width - ClientWidth
    end else begin
      ClientYmod := 0;
      ClientXmod := 0
    end;

  if Assigned (FOnModifyClient) then
    FOnModifyClient (Self, ClientXmod, ClientYmod);

  // Rect is the current form size (not client area), we can
  // play with this TRect to change the form size in anyway.
  with Rect do
  // the Side contains a value indicating what is to be adjusted
  // actually uses the WMSZ_XXXX values which is the side being pulled
  // or the corner being pulled (eight choices).
  // So calculate Right, Top, Bottom or Left values.  The combinations set
  // here are based on what feels "right" for me, some experimentation
  // for your feel may be necessary.  If you change this behaviour, you may
  // need to change the saXXXX constants as well.
    case Side of
      WMSZ_BottomRight,
      WMSZ_Bottom : Right := Left + (Bottom - Top - ClientYmod) * AspectX div AspectY + ClientXmod;

      WMSZ_BottomLeft,
      WMSZ_Right  : Bottom := Top + (Right - Left - ClientXmod)* AspectY div AspectX + ClientYmod;

      WMSZ_TopRight,
      WMSZ_Left   : Top := Bottom - (Right - Left - ClientXmod) * AspectY div AspectX - ClientYmod;

      WMSZ_TopLeft,
      WMSZ_Top    : Left := Right - (Bottom - Top - ClientYmod) * AspectX div AspectY - ClientXmod
    end
end;

procedure TAspect.NewDefWndProc (var Msg : TMessage);
var
  ClientXmod,
  ClientYmod : integer;
  FormRect : TRect;
  Xi,
  Yi : integer;
begin
  if FActive then  // if we are on then ...
    with Msg do
    begin
  // if the message is sizing
      if Msg = WM_Sizing then
      begin

    // lParam points to a TRect which is the current form size
    // the wParam contains a value indicating what is being pulled
       EstablishAspectRatio (wParam, PRect (lParam)^);

       if Assigned (FOnBeforeResizing) then
         FOnBeforeResizing (Self, PRect (lParam)^);

        Result := 0; // done
        exit
      end;

// is the form being zoomed?
      if Msg = WM_GetMinMaxInfo then
      begin

  // get the form/client size adjustment
        if FClient then
          with Owner as TForm do
          begin
            ClientYmod := Height - ClientHeight;
            ClientXMod := Width - ClientWidth
          end else begin
            ClientYmod := 0;
            ClientXmod := 0
          end;
        if Assigned (FOnModifyClient) then
          FOnModifyClient (Self, ClientXmod, ClientYmod);

   // if so calculate a size, taking into account strange screen sizes
        with PMinMaxInfo (lParam)^.ptMaxSize do
        begin
          Xi := X;
          Yi := X * FAspectY div FAspectX;
          if Yi > GetSystemMetrics (SM_CYScreen) then
          begin
            Yi := Y;
            Xi := Y * FAspectX div FAspectY
          end;

// make any adjustment for client sizing and call the before event
          FormRect := Rect (0, 0, Xi + ClientXmod, Yi + ClientYMod);
          if Assigned (FOnBeforeResizing) then
            FOnBeforeResizing (Self, FormRect);

// put the final value into the ptMaxSize record
          with FormRect do
          begin
            X := Right - Left;
            Y := Bottom - Top
          end
        end;
        Result := 0; // done
        exit
      end
  end;

// otherwise call the original WndProc
  with Msg do
    Result := CallWindowProc (FOldDefWndProc, FParent, Msg, wParam, lParam)
end;

// I like it on the System Page, you might like to put it on Win95 since it's
// a win 32 only component
procedure Register;
begin
  RegisterComponents (LoadStr(srSystem), [TAspect]);
end;

end.

